/**
	File containing QuestAgent JavaScript classes in <i>questagent</i> namespace.
	Copyright (c) 1997-2010 JOBJECTS. All rights reserved.
	@file questagent.js
*/

/**
	Namespace <i>questagent</i> encapsulates JavaScript classes that represent a base
	for QuestAgent search interface integration.
	@namespace questagent
*/

var questagent;

if (typeof questagent.id == "undefined")
{
(function() {

	if( !questagent.util ) {
		var err = 'Error: Script file "qautil.js" must be included before "questagent.js".';
		alert( err );
		throw err;
	}

	if( !YAHOO ) {
		var err = 'Error: Yahoo User Interface Library (YUI) must be included before "questagent.js".';
		alert( err );
		throw err;
	}

	YAHOO.namespace("jobjects.questagent");

	// shortcuts for YUI components
	var Overlay = YAHOO.widget.Overlay;
	var Event = YAHOO.util.Event;
	var Dom = YAHOO.util.Dom;

	var log = questagent.util.log;

	questagent.version = "8.1.3.5829";
	questagent.log = log;

	var applicationStartDate = new Date();
	var uniqueId = "qa" + applicationStartDate.getTime() + "_" + Math.floor(Math.random() * 100000000);

	log.info('Running in browser: ' + navigator.userAgent);

	questagent.id = uniqueId;
	log.info('Instantiated questagent with id "' + questagent.id +
		'" in window "' + window.name + '" (' + window.location + ')');

	var iframeRefName = questagent.id + '_ref';


	/** Search Applet */
	// ======================== Search Applet Wrapper starts ===================

	(function() {

		/**
		Class that initializes and interacts with QuestAgent Java applet.
		In most cases you will not need to customize this class.
		@class questagent.SearchAppletWrapper
		*/
		/**
		Initializes search applet wrapper class.
		@constructor SearchAppletWrapper
		@param {object} config - configuration object
		@param {object} model - search interface model instance
		@param {string} html - HTML segment that should be used for initialization of QuestAgent Java applet.
		*/
		var SearchAppletWrapper = function(cfg,model,appletHTML) {

			this.searchInProgress = false;

			log.info("Running QuestAgent scripts version " + questagent.version);

			log.debug( "SearchAppletWrapper()" );

			this.config = cfg;

			log.debug( "Create evenets." );
			// Create custom events. Scope in which even is fired is always search interface model.
			this.eventInitializingApplet = new YAHOO.util.CustomEvent(
				"InitializingApplet", model, this, YAHOO.util.CustomEvent.FLAT);
			this.eventAppletInitialized = new YAHOO.util.CustomEvent(
				"AppletInitialized", model, this, YAHOO.util.CustomEvent.FLAT);
			this.eventSearchStarting = new YAHOO.util.CustomEvent(
				"SearchStarting", model, true, YAHOO.util.CustomEvent.FLAT);
			this.eventSearchCompleted = new YAHOO.util.CustomEvent(
				"SearchCompleted", model, true, YAHOO.util.CustomEvent.FLAT);


			if( !qagentConfig.containerDocument )
				qagentConfig.containerDocument = document;

			var appletContainer = cfg.containerElement;
			if( !appletContainer ) {
				if( cfg.containerDocument && cfg.containerElementName ) {
					appletContainer = cfg.containerDocument.getElementById(cfg.containerElementName);
					if( !appletContainer ) {
						log.fatal( "Can't find search applet container element: " + cfg.containerElementName +
							' in ' + cfg.containerDocument.location );
						return false;
					}
					log.debug( "Found container element for search applet." );
					//appletContainer.style.display = "none"; // Firefox will not load Java applet when display is set to none!
				}
				else {
					// not found container element
					log.info( "Not found applet container element by provided parameters. Using document body." );
					var body = qagentConfig.containerDocument.getElementsByTagName("body")[0];
					var d = document.createElement("div");
					YAHOO.util.Dom.setStyle(d, 'position', 'absolute');
					YAHOO.util.Dom.setStyle(d, 'top', '-200px');
					YAHOO.util.Dom.setStyle(d, 'left', '-200px');
					appletContainer = body.appendChild(d);
				}
			}

			// notify model that wrapper events have been initialized
			model.setAppletWrapper( this );

			// initialization starting
			this.eventInitializingApplet.fire();

			log.debug('Applet init HTML: ' + appletHTML);

			appletContainer.innerHTML = appletHTML;

			this.config.iframeRefName = iframeRefName;

			// another hidden frame used for obtaining reference to qa model
			var iframeRefDiv = document.createElement("div");
			YAHOO.util.Dom.setStyle(iframeRefDiv, 'position', 'absolute');
			YAHOO.util.Dom.setStyle(iframeRefDiv, 'top', '-200px');
			YAHOO.util.Dom.setStyle(iframeRefDiv, 'left', '-200px');
			appletContainer.appendChild(iframeRefDiv);
			iframeRefDiv.innerHTML = "<iframe id='" + iframeRefName + "' name='" +
				questagent.id +	"_ref' width='0' height='0' frameborder='0'" +
				"scrolling='no' src='" + questagent.util.getQAHome(this.config) + "empty.htm'></iframe>";

			var scriptsPathFinder = function()
			{
				// Scripts path is needed as IE and FF in a different way resolve relative paths from dynamically created documents
				try {
					scriptsPathFinder.config.qaHomeURL = questagent.util.findQAHomeURL(iframeRefName);
				}
				catch( e ) {
					// exception may be thrown by findScriptsPath if iframe is not loaded yet,
					// so we schedule one more execution...
					window.setTimeout( scriptsPathFinder, 200 );
					return;
				}

				log.debug('Determined QA Home URL as: ' + scriptsPathFinder.config.qaHomeURL);

				if( window.qaAutoConfig ) {
					window.qaAutoConfig.qaHomeURL = scriptsPathFinder.config.qaHomeURL;
					log.info('Set qaAutoConfig.qaHomeURL: ' + window.qaAutoConfig.qaHomeURL);
				}
			};
			scriptsPathFinder.config = this.config;

			// as soon as reference iframe is loaded, collect absolute path to scripts folder
			Event.onAvailable(iframeRefName, scriptsPathFinder, scriptsPathFinder, true );

			return true;
		}

		/**
		Invoked after QuestAgent applet has started.
		Params.applet references QuestAgent applet element.
		@function onAppletStarted
		@param {object} params - object containing initialization information.
		*/
		SearchAppletWrapper.prototype.onAppletStarted = function(param) {
			// NOTE: never try to include applet ref in print as it breaks in Safari 3
			log.debug("SearchAppletWrapper.onAppletStarted");

			this.applet = param.applet;

			log.debug("Loaded search applet: " + this.applet.getClass() );
			log.debug("Available collections: " + this.applet.getCollectionCount() );

			// determine base URL (where jar is located)
			this.config.baseURL = this.applet.getDocumentBase( "" );
			log.info("Determined base URL: " + this.config.baseURL );

			this.eventAppletInitialized.fire({
				applet: this.applet,
				baseURL: this.config.baseURL
			} );
		}

		/**
		Starts search. Results will be delivered via callback method.
		Callback can be either function or object that defined "onSearchResults" function.
		@function search
		@param {object} searchParams - object encapsulating search parameters
		*/
		SearchAppletWrapper.prototype.search = function(searchParams) {
			log.debug("SearchAppletWrapper.search");

			try
			{
				this.eventSearchStarting.fire(searchParams);

				if( !this.applet )
				{
					var result = new Object();
					result.hasError = true;
					result.errorMessage = "Applet is not available. Is wrapper object initialized?";

					log.fatal(result.errorMessage);

					this.eventSearchCompleted.fire(result);
					return;
				}

				if( this.searchInProgress )
				{
					var result = new Object();
					result.hasError = true;
					result.errorMessage = "Search is already running.";

					log.error(result.errorMessage);

					// in this case we don't invoke callback as it's expected that another running thread
					// will actually deliver real results
					// this.eventSearchCompleted.fire(result);
					return;
				}

				this.verifySearchParams( searchParams );

				this.searchInProgress = true;
				this.timeSearchStarted = new Date();

				// save search params for later
				this.searchParams = searchParams;

				if( searchParams.colls ) {
					log.debug("Received collection list of type " + (typeof searchParams.colls));

					this.applet.unselectAllCollections();
					if( (typeof searchParams.colls) == 'string' ) {
						log.debug('Select collection: ' + searchParams.colls);
						if( searchParams.colls == '[all]' )
							this.applet.selectAllCollections();
						else
							this.applet.selectCollection( searchParams.colls );
					}
					else {
						for( i = 0; i < searchParams.colls.length; ++i ) {
							log.debug('Select collection: ' + searchParams.colls[i]);
							if( searchParams.colls[i] == '[all]' ) // make sense only if the only item in array
								this.applet.selectAllCollections();
							else
								this.applet.selectCollection( searchParams.colls[i] );
						}
						if( searchParams.colls.length == 0 )
							this.applet.selectAllCollections();
					}
				}
				else {
					log.info("Selecting all available collections for search as none is explicitly specified in request.");
					this.applet.selectAllCollections();
				}

				log.debug( "Start hit: " + searchParams.startHit );
				this.applet.setStartHit( searchParams.startHit );

				log.debug( "Hits per page (maxHits): " + searchParams.hitsPerPage );
				this.applet.setMaxHits( searchParams.hitsPerPage );

				log.info("Query: " + searchParams.query);

				log.debug("Starting search.");
				var hitCount = this.applet.search( searchParams.query );

				if( hitCount == -2 ) {
					log.debug( "Search started. Callback/event from search applet will be used to deliver results." );
				}
				else {
					log.debug( "Search applet returned results right away." );
					this.onSearchCompleted(hitCount);
				}
			}
			catch( e ) {
				questagent.util.log.error(e);
			}
		}

		/*
		Verifies search parameters and sets missing search params to defaults.
		@function verifySearchParams
		@param {object} searchParams - object encapsulating search parameters
		*/
		SearchAppletWrapper.prototype.verifySearchParams = function(searchParams) {
			log.debug( "SearchAppletWrapper.verifySearchParams" );

			if( !searchParams.startHit )
				searchParams.startHit  = 1;

			if( !searchParams.hitsPerPage )
				searchParams.hitsPerPage = this.config.hitsPerPage;

			// FIXME check what's really used of fields by default
			if( !searchParams.fields )
				searchParams.fields = new Array("title", "(title_hl)", "(summary_hl)", "(kic)", "description" );
		}

		/**
		Receives internal event that search has completed, packs resulting object and passes to model.
		@function onSearchCompleted
		@param {int} hitCount - number of hits
		*/
		SearchAppletWrapper.prototype.onSearchCompleted = function(hitCount) {
			log.debug("SearchAppletWrapper.onSearchCompleted");

			if( !this.searchInProgress ) {
				log.fatal("Illegal state. Search was not in progress by internal status flag.");
				return;
			}

			log.debug("Packing search results object");

			var result = new Object();

			result.searchParams = this.searchParams;
			this.searchParams = null;
			result.timeSearchStarted = this.timeSearchStarted;
			this.timeSearchStarted = null;
			result.timeSearchCompleted = new Date();

			log.info("Search completed in " +
				(result.timeSearchCompleted.getTime() - result.timeSearchStarted.getTime()) + "ms");

			result.hitCount = this.applet.getHitCount();
			log.debug("Hit count: " + result.hitCount);

			result.totalHitCount = this.applet.getTotalHitCount();
			result.firstHit = this.applet.getFirstHit();
			result.lastHit = this.applet.getLastHit();
			log.debug("Total hits " + result.totalHitCount + " collecting " + result.firstHit + "-" + result.lastHit );

			result.hits = new Array();
			for(i=result.firstHit; result.hitCount > 0 && i<=result.lastHit; i++)
			{
				var hit = new Object();
				result.hits[i] = hit;
				hit.index = i;

				hit.location = "" + this.applet.getHitLocation( i );

				// extract requested fields
				var fields = new Array();
				if( result.searchParams.fields ) {
					for(j=0; j<result.searchParams.fields.length; j++) {
						var fieldName = result.searchParams.fields[j];
						var data = this.applet.getHitField( hit.index, fieldName );
						if( data ) { // if found, add it to the map converting to JavaScript string
							fields[ fieldName ] = "" + data;
							//log.debug( "Found hit field data: " + data );
						}
						else {
							log.debug( "Not found hit field '" + fieldName + "' in " + hit.location  );
						}
					}
				}
				hit.fields = fields;

				log.debug( i + ". " + hit.location );

				hit.length = this.applet.getHitField( i, "length" );
				if( hit.length ) {
					var len = Math.round( eval( "" + hit.length ) / 1024 );
					if( len == 0 )
						len = 1;
					hit.lengthKB = len;
				}

				hit.score = this.applet.getHitRelevance( i );
			}

			// pack log messages
			result.messages = new Array();
			result.messagesCount = this.applet.getLogMessagesCount();
			for(i = 0; i < result.messagesCount; i++) {
				result.messages[i] = new Object();
				result.messages[i].text = this.applet.getLogMessage(i);
				result.messages[i].type = this.applet.getLogMessageType(i); // DEBUG = 1; INFO = 2; WARN = 3; ERROR = 4; FATAL = 5
				log.info("Search log message: " + result.messages[i].text + " (type:" + result.messages[i].type + ")");
			}

			result.hasError = false;
			// TODO ??? what?

			result.timeResultsCollected = new Date();

			log.info("Results collected in " +
				(result.timeResultsCollected.getTime() - result.timeSearchCompleted.getTime()) + "ms");

			this.eventSearchCompleted.fire(result);

			this.searchInProgress = false;
		}

		questagent.SearchAppletWrapper = SearchAppletWrapper;

	})();

	// ======================== QuestAgent models start ===================

	(function() {

		/**
		Class that contains QuestAgent configuration defaults and messages.
		@class questagent.QAConfigDefaults
		*/
		var QAConfigDefaults = function() {

			// OPTIONAL (default is 'qagent8/qagent.prm')
			//this.paramFile = 'sample01/qagent.prm';

			// OPTIONAL (otherwise finds the largest frame)
			//this.resultsTargetFrameName = "center";
			//this.documentTargetFrameName = "center";

			this.highlightingEnabled = true;

			// set in super class or in validator function?...
			this.hitsPagesTitle = "More hits:";
			this.prevHitsPageTitle = "&lt;";
			this.nextHitsPageTitle = "&gt;";
			this.prevLinksPageTitle = "[...]";
			this.nextLinksPageTitle = "[...]";

			this.msgInitializingQA = "Initializing search engine";
			this.msgAllCollections = "[all collections]";
			this.msgFormLinkTitle = "Click for full text search";
			this.msgAltFormTitle = "Full text search";
			this.msgAltFormButtonTitle = "Find";

			this.msgPleaseWaitTitle = "Please wait...";
			this.msgSearchInProgress = "Search is in progress.";
			this.msgHighlightingInProgress = "Highlighting search terms in document...";
			this.msgInitializingSearchEngine = "Initializing QuestAgent search engine.";
			this.imgWaitBig = "loader-indicator-big.gif";
			this.imgWaitSmall = "loader-indicator-small.gif";

			this.hitsPerPage = 10;
			this.pageListSize = 10;

			this.popupWidth = 620;
			this.popupHeight = 550;

			this.showBaseURL = false;
		};
		questagent.QAConfigDefaults = QAConfigDefaults;


		/**
		Base class for QuestAgent search interfaces that use HTML presentation of search form and results.
		@class questagent.HtmlJSBasedModel
		*/
		var HtmlJSBasedModel = function() {
		};

		/**
		Initializes this search interface model.
		@function init
		@param {object} config - configuration object
		*/
		HtmlJSBasedModel.prototype.init = function(cfg) {
			this.config = cfg;
			this.log = log;
			this.lastSearchResults = null;
			this.lastHighlightedDocIndex = null;

			log.debug('Initializing HtmlJSBasedModel for host window: ' + window.name);

			if(window.qagentModel) {
				log.fatal("Only one search interface model can be defined per window/frame. " +
					"Found " + window.qagentModel + " so can't define " + this.id );
				return false;
			}
			window.qagentModel = this;

			var config = this.config; // this variable will be referenced in 'onLoad' handler bellow as 'this' can't be used

			Event.onDOMReady( function()
			{
				log.debug("QA home window loaded. Starting init...");

				var form = questagent.util.attachSearchFormHandlers( window, config );

				if( form ) {
					log.debug("Disabling search form");
					questagent.util.formSetEnabled(form, false);

					if( config.queryTextFieldId && config.msgInitializingQA ) {
						var queryField = window.document.getElementById(config.queryTextFieldId);
						if( queryField )
							queryField.value = config.msgInitializingQA;
					}
				}

				// Starting applet init with a slight delay as FF (and possibly other browser)
				// may show document status as "loading".
				window.setTimeout( "questagent.initSearchApplet()", 100);
			}, window, true );
			log.debug("Added 'onDOMReady' event listener.");

			log.info("Interface model initialized");
		}

		/**
		Connects this search interface model with applet wrapper.
		@function setAppletWrapper
		@param {object} wrapper - applet wrapper object
		*/
		HtmlJSBasedModel.prototype.setAppletWrapper = function(wrapper) {
			this.appletWrapper = wrapper;

			log.debug('Subscribing to applet events.');

			wrapper.eventInitializingApplet.subscribe( this.onSearchEngineInitStarts, this );
			wrapper.eventAppletInitialized.subscribe( this.onSearchAppletInitialized, this );
			wrapper.eventSearchCompleted.subscribe( this.onSearchResults, this );
		}

		/**
		Returns HTML segment that will be used to initialize search applet.
		@function getAppletInitHTML
		@returns string with HTML segment
		*/
		HtmlJSBasedModel.prototype.getAppletInitHTML = function() {

			log.debug("In 'iframe' loading Java test and then search applet...");

			// 1) Adding an iframe using the DOM would be preferable, but it doesn't work in IE5 on Windows,
			// or in Konqueror prior to version 3.5 - in Konqueror it creates the iframe fine but couldn't
			// find a way to obtain the iframe's window object
			// 2) IE will not load applet inserted with OBJECT tag if width/height is set to zero.
			return "<iframe id='" + uniqueId + "_Applet' name='" + uniqueId +
				"_Applet' width='1' height='1' frameborder='0'" +
				// Loading Java test that will load search applet. An alternative is to load "lcapplet.htm" right away.
				"scrolling='no' src='" + questagent.util.getQAHome(this.config) + "testjava.htm'></iframe>";
		}

		/**
		Event handler function invoked when search engine is starting with initialization.
		@function onSearchEngineInitStarts
		*/
		HtmlJSBasedModel.prototype.onSearchEngineInitStarts = function() {
			var showInitOverlay = window.qagentBootstrap ? window.qagentBootstrap.showInitOverlay : true;
			if( showInitOverlay )
				questagent.util.showEngineInitOverlay(window, this.config);
		}

		/**
		Event handler function invoked when search applet/engine is initialized.
		@param {object} params - object containing initialization parameters. Field params.applet contains reference to applet element.
		@function onSearchAppletInitialized
		*/
		HtmlJSBasedModel.prototype.onSearchAppletInitialized = function(args) {
			log.debug("HtmlJSBasedModel.onSearchAppletInitialized");

			/* // reference to applet methods still might not be valid (Java 1.4)
			if(args.applet.getClass)
				log.debug("Applet class " + args.applet.getClass() );
				*/

			var form = window.document.getElementById(this.config.searchFormId);

			var ifr = qagentConfig.containerDocument.getElementById(uniqueId + '_Applet');
			if( ifr ) {
				log.warn('Hiding applet iframe.');
				ifr.style.width='0';
				ifr.style.height='0';
			}
			else {
				log.warn('Not found iframe by ID ' + uniqueId + ' to hide it.');
			}

			if( YAHOO.jobjects.questagent.initializingMessage )
				YAHOO.jobjects.questagent.initializingMessage.hide();

			if( this.config.collectionSelectId )
			{
				var collectionSelect = window.document.getElementById(this.config.collectionSelectId);
				if( collectionSelect )
				{
					questagent.util.populateCollectionList(
						collectionSelect, this.appletWrapper.applet, this.config.msgAllCollections );
				}
			}


			var queryField = window.document.getElementById(this.config.queryTextFieldId);
			if( queryField ) {
				queryField.value = "";

				// is search already triggered?
				if( this.config.params && this.config.params.query ) {
					queryField.value = this.config.params.query;
					// invoke startSearch() if available in context
					window.setTimeout( 'if( typeof startSearch == "function" ) startSearch();', 100);
				}
			}
			else
				log.debug('No form field found: ' + this.config.queryTextFieldId);

			log.debug("Enabling search form: " + form);
			if( form )
				questagent.util.formSetEnabled(form, true);
		}

		/**
		This function starts search.
		Search process will run asynchronously and results will be delivered to <i>onSearchResults</i> function.
		@param {object} searchParams - search parameters
		@param {int} startHit - start hit; will default to  if not defined.
		@function search
		*/
		HtmlJSBasedModel.prototype.search = function(searchParams,startHit) {
			log.debug("in HtmlJSBasedModel.search()");

			if( searchParams )
				this.searchParams = searchParams;
			else
				log.debug("Reusing old search parameters (paging)");

			if( !this.searchParams )
			{
				log.error("Search parameters are not specified.");
				// FIXME report error to user
				return;
			}

			var waitFrameRef = this.getWaitPanelTargetFrameRef()
			if( waitFrameRef ) {
				log.debug('Found target for "wait" panel: ' + waitFrameRef.name);
				questagent.util.showPleaseWaitPanel(
					this.config.msgSearchInProgress, waitFrameRef, this.config );
			}

			if( startHit ) {
				this.searchParams.startHit = startHit;
			}

			this.appletWrapper.search(this.searchParams);
		}

		/**
		This function received starts results.
		@param {object} results - search results
		@function onSearchResults
		*/
		HtmlJSBasedModel.prototype.onSearchResults = function(result) {
			log.debug("HtmlJSBasedModel.onSearchResults");

			this.hlInfo = null;
			this.lastSearchResults = result;

			// build HTML segment with results and save with results
			result.html = this.buildResultsHTML(result);

			//log.debug( "Results HTML: " + result.html);

			var targetFrameName = this.getResultsTargetFrameName();
			log.debug("Open results page in: " + targetFrameName);

			this.config.windowWithResults = window.open(
				questagent.util.getQAHome(this.config) + "results.htm", targetFrameName);
		}

		/**
		Builds HTML segment specifying number of hits.
		@function buildResultsNotice
		@param {object} results - search results
		@returns string with HTML segment
		*/
		HtmlJSBasedModel.prototype.buildResultsNotice = function(result) {
			log.debug("HtmlJSBasedModel.buildResultsNotice");

			if( result.totalHitCount > 1 ) {
				return ("Results: <b>" + result.firstHit + " - " + result.lastHit + "</b> of total <b>" + result.totalHitCount + "</b>.");
			}
			else if( result.totalHitCount == 1 ) {
				return "Results: 1 document found.";
			}
			return "";
		}

		/**
		Builds HTML segment for search results pagination.
		@function buildHitsPagesLinks
		@param {object} results - search results
		@returns string with HTML segment
		*/
		HtmlJSBasedModel.prototype.buildHitsPagesLinks = function(result) {
			log.debug("HtmlJSBasedModel.buildHitsPagesLinks");

			var buf = "";
			var pageSize = this.config.hitsPerPage;
			var pageListSize = this.config.pageListSize; // TODO add function to set defaults to config

			// first hit on the current page
			var currentStartHit = result.firstHit;

			// current page
			var currentPage = Math.ceil( currentStartHit / pageSize );

			// last results page
			var lastPage = Math.floor( result.totalHitCount / pageSize );

			if( result.totalHitCount % pageSize != 0 )
			{
				++lastPage;
			}

			if( lastPage > 1 )
			{
				// first page in the page links list
				var startPage = ( Math.floor( ( currentPage - 1 ) / pageListSize ) ) * pageListSize + 1;

				// last page in the page links list
				var endPage = Math.min( lastPage, startPage + pageListSize - 1 );

				if( this.config.hitsPagesTitle != null )
				{
					buf += this.config.hitsPagesTitle;
					buf += "&nbsp;&nbsp;";
				}

				if( currentPage > 1 )
				{
					buf += "<b>";
					buf += this.buildResultPageLink( currentPage - 1, pageSize, this.config.prevHitsPageTitle );
					buf += "</b>&nbsp;&nbsp;";
				}

				if( startPage > 1 )
				{
					buf += this.buildResultPageLink( startPage - 1, pageSize, this.config.prevLinksPageTitle );
					buf += "&nbsp;&nbsp;";
				}

				for( var i = startPage; i <= endPage; i ++ )
				{
					if( i != currentPage )
						buf += this.buildResultPageLink( i, pageSize, i );
					else
						buf += "<b>" + i + "</b>";

					if( i != endPage )
						buf += "&nbsp;&nbsp;";
				}

				if( endPage < lastPage )
				{
					buf += "&nbsp;&nbsp;";
					buf += this.buildResultPageLink( endPage + 1, pageSize, this.config.nextLinksPageTitle );
				}

				if( currentPage < lastPage )
				{
					buf += "&nbsp;&nbsp;<b>";
					buf += this.buildResultPageLink( currentPage + 1, pageSize, this.config.nextHitsPageTitle );
					buf += "</b>";
				}
			}
			return buf;
		}

		/**
		Builds HTML segment with link that triggers paginated search.
		@function buildResultPageLink
		@param {int} page - results page for which link is requested
		@param {int} pageSize - number of results per page
		@param {string} linkText - link text
		@returns string with HTML segment
		*/
		HtmlJSBasedModel.prototype.buildResultPageLink = function( page, pageSize, pageText )
		{
			var sh = (page-1)*pageSize + 1;
			var link = '<a href="javascript:';
			link += 'questagent.util.getSearchInterfaceModel().search(null,' + sh + ');"';
			link += '>' + pageText + '</a>';
			return link;
		}

		/**
		Creates HTML for search results page.
		Relies on other helper fucntions for specific page elements.
		@function buildResultsHTML
		@param {object} results - search results
		@returns string with HTML segment
		*/
		HtmlJSBasedModel.prototype.buildResultsHTML = function(result) {
			log.debug("HtmlJSBasedModel.buildResultsHTML");

			var html = [];

			html.push('<div class="qaresnotice">');
			html.push(this.buildResultsNotice(result));
			html.push('</div>');

			if( result.totalHitCount > 0 ) { // list results
				html.push('<div class="qahits">');
				for(i=result.firstHit; i<=result.lastHit; i++)
				{
					var hit = result.hits[i];
					//if( hit == null )
						//break;
					html.push('<div class="qahit">');
					html.push('<h2 class="qahith">');
					//html.push(hit.index);
					//html.push('. ');
					html.push('<a href="');
					html.push(hit.location);
						// TODO maybe can allow highlighting in "new window" if we target hldoc.htm ?
					html.push('" target="');
					html.push(this.getDocumentTargetFrameName());
					html.push('" ');
					html.push('onclick="questagent.util.getSearchInterfaceModel().showDocument(');
					html.push(hit.index);
					html.push('); return false;">');
					html.push(hit.fields["(title_hl)"]);
					html.push('</a>');
					html.push('</h2>');

					html.push('<table border="0" cellpadding="0" cellspacing="0">');
					html.push('<tbody><tr><td class="j"><font size="-1">');

					html.push(hit.fields["(summary_hl)"]);

					html.push('<br><span class="a">');

					// show file path with link that opens document without highlighting
					html.push('<a href="');
					html.push(hit.location);
					html.push('" ');
					html.push('target="');
					html.push(this.getDocumentTargetFrameName());
					html.push('">');
					if( this.config.showBaseURL )
						html.push(this.config.baseURL);
					html.push(hit.location);
					html.push('</a>');

					// show file size
					if( hit.lengthKB ) {
		 				html.push(' - ');
						html.push(hit.lengthKB);
		 				html.push('kb');
					}

					//html.push(' - Score: ' + hit.score );

					html.push('</span></font></td></tr></tbody></table>');
					html.push('</div>');
				}
				html.push('</div>');

				html.push('<p>&nbsp;</p>');

				var pagesBuf = this.buildHitsPagesLinks( result );
				if( pagesBuf.length > 0 )
				{
					html.push('<div class="qarespages">');
					//html.push('<table align="center"><tbody><tr align="center" style="text-align: center;">');
					html.push('<center>');
					html.push(pagesBuf);
					html.push('</center>');
					//html.push('</tr></tbody></table>');
					html.push('</div>');
				}
			}
			else { // nothing found

				html.push('<p>Your search did not match any documents.</p>');

				html.push('<p>Suggestions:<ul>');
				html.push('<li>Make sure all words are spelled correctly.</li>');
				html.push('<li>Try different keywords.</li>');
				html.push('<li>Try more general keywords.</li>');
				html.push('</ul></p>');
			}

			if( result.messages ) {
				html.push('<div class="qamsgs">');
				for(i = 0; i < result.messages.length; i++) {
					var msg = result.messages[i];
					if( msg ) {
						html.push('<div class="qamsg');
						html.push(''+msg.type); // DEBUG = 1; INFO = 2; WARN = 3; ERROR = 4; FATAL
						//html.push('">--');
						//html.push(''+msg.text);
						//html.push('--</div>');
					}
				}
				html.push('</div>');
			}

			html.push('<p>&nbsp;</p><p>&nbsp;</p><hr class="f" /><div align="center"><font size="-2">');
			// REMOVAL OR MODIFICATION OF THE MESSAGE BELLOW IS IN VIOLATION OF THE LICENSE AGREEMENT - DO NOT TOUCH!
			html.push('Generated by QuestAgent<br/>Search technology from JObjects (<a href="http://www.jobjects.com" target="_blank">http://www.jobjects.com</a>)');
			html.push('</font></div>');

			return html.join('');
		}

		/**
		This function is invoked when user clicks on a document link in search results.
		Starts document highlighting process if highlighting is possible/supported.
		@function showDocument
		@param {int} hitIndex - index of requested document in hits array.
		@param {string} location - URI of document location (optional)
		*/
		HtmlJSBasedModel.prototype.showDocument = function( hitIndex, location, cwin ) {
			log.debug( "showDocument() " + hitIndex + ", " + location + ", " + cwin );

			var loc = location || this.lastSearchResults.hits[hitIndex].location;

			log.debug( "Show hit " + hitIndex + ": " + loc );

			var hlopt = this.getHighlightingOptions(loc);
			hlopt.docLocation = loc;
			hlopt.hitIndex = hitIndex;
			this.hlInfo = hlopt;

			if( hlopt.enabled && questagent.canHighlight(loc) )
			{
				questagent.util.showPleaseWaitPanel(
					this.config.msgHighlightingInProgress, this.config.windowWithResults, this.config );

				this.appletWrapper.applet.setHighlightingTags(
					hlopt.hlTagBefore, hlopt.hlTagAfter );
				this.appletWrapper.applet.setNavigationButtons(
					hlopt.buttonPreviousAnchor, hlopt.buttonNextAnchor );
				this.appletWrapper.applet.setNavigationLinkScheme(
					hlopt.navOpenTagStart, hlopt.navOpenTagEnd, hlopt.navCloseTag );

				hlopt.hlDocBase = this.appletWrapper.applet.getDocumentBase( loc );

				log.debug( "Base of highlighted document: " + hlopt.hlDocBase );

				if( hlopt.useCallback )
				{
					log.debug( "Highlight using callback/event page delivery" );

					this.appletWrapper.applet.highlightDocumentCB( hitIndex, hlopt.addNavTags, "OnHighlightingDone" );
				}
				else
				{
					log.debug( "Highlight directly (without using callback)" );

					var page = this.appletWrapper.applet.highlightDocument( hitIndex, hlopt.addNavTags );
					this.handleHighlightedDocument( page );
				}

			}
			else {
				log.debug( "Highlighting is not supported or enabled. Show document as is." );
				this.showDocumentNoHL(loc);
			}
		}

		/**
		Returns parameters used by document highlighter.
		@function getHighlightingOptions
		@param {string} location - URI of document location
		@returns object with highlighting options
		*/
		HtmlJSBasedModel.prototype.getHighlightingOptions = function( loc )
		{
			var hlopt = new Object();

			hlopt.enabled = this.config.highlightingEnabled;
			hlopt.addNavTags = true;

			// hl navigation doesn't work in MacOS X Safari
			if( questagent.util.UA.bIsSafari )
				hlopt.addNavTags = false;

			// callback required with NS 6/7 running from a web site
			if( questagent.util.UA.bIsGecko && !questagent.util.UA.bIsSafari )
				hlopt.useCallback = true;
			else // we don't use callbacks with IE (some versions crash on second highlighted doc)
				hlopt.useCallback = false;

			// FIXME get options from coockies ?

			// Tags used to mark highlighted terms
			hlopt.hlTagBefore = "<b style=\"background-color: #ffff66; color: black\">"; // FIXME use external CSS?
			hlopt.hlTagAfter = "</b>";

			// Tags for buttons: "previous term" and "next-term"
			hlopt.buttonPreviousAnchor = "<font size=1>&lt;</font>";
			hlopt.buttonNextAnchor = "<font size=1>&gt;</font>";

			// Open tag for navigation link (anchor ID is inserted between the start and end tag)
			hlopt.navOpenTagStart = '<a href="#" onclick="window.location.hash=\'#qa_anchor_';
			hlopt.navOpenTagEnd = '\'; return false;">';

			// Close tag for navigation link
			hlopt.navCloseTag = "</a>";

			return hlopt;
		}

		/**
		This function receives highlighted page from QuestAgent search applet and
		further calls <i>showDocumentHL</i> if highlighting was successful.
		@function handleHighlightedDocument
		@param {string} page - highlighted page
		*/
		HtmlJSBasedModel.prototype.handleHighlightedDocument = function(hlPage)
		{
			questagent.util.hidePleaseWaitPanel(this.config.windowWithResults);

			// check is returned content actually error message
			if( hlPage.indexOf( "qagent_hl_msg:" ) == 0 )
			{
				var message = hlPage.substring( 14 ); // extract error message
				log.debug("Error on highlighting: " + message);
				this.handleHighlightingError(message,this.hlInfo);
				return;
			}

			log.debug("Received highlighted document.");

			this.hlInfo.docHighlighted = '<base href="' + this.hlInfo.hlDocBase + '">' + hlPage;
			this.hlInfo.currentAnchor = "qa_anchor_1";

			this.showDocumentHL( this.hlInfo );
		}

		/**
		Invoked if error happened during highlighting.
		By default shows alert message and shows document without highlighting.
		@function handleHighlightingError
		@param {string} message - error message
		@param {object} options - highlightiong options
		*/
		HtmlJSBasedModel.prototype.handleHighlightingError = function( message, hlInfo )
		{
			alert( message ); // show message
			this.showDocumentNoHL( this.hlInfo.docLocation ); // show without highlighting
		}

		/**
		Returns reference to frame where document should open.
		@function getDocumentTargetFrameRef
		@returns frame reference or null if not found
		*/
		HtmlJSBasedModel.prototype.getDocumentTargetFrameRef = function()
		{
			var f;
			if( parent )
				f = questagent.util.findLargestFrameDeep(parent);
			else
				f = questagent.util.findLargestFrameDeep(window);
			if( f && f.frame )
				return f.frame;
			return null;
		}

		/**
		Returns name of frame where document should open.
		If name of target window was not provided in configuration parameters, it will return
		name of the largest frame.
		@function getDocumentTargetFrameName
		@returns string with target frame/window name
		*/
		HtmlJSBasedModel.prototype.getDocumentTargetFrameName = function()
		{
			if( !this.config.documentTargetFrameName )
			{
				var f;
				if( parent )
					f = questagent.util.findLargestFrameDeep(parent);
				else
					f = questagent.util.findLargestFrameDeep(window);
				if( f && f.name )
					this.config.documentTargetFrameName = f.name;
			}
			return this.config.documentTargetFrameName;
		}

		/**
		Returns name of frame where search results should be shown.
		If name of target window was not provided in configuration parameters, it will return
		name of the largest frame.
		@function getResultsTargetFrameName
		@returns string with target frame/window name
		*/
		HtmlJSBasedModel.prototype.getResultsTargetFrameName = function()
		{
			if( !this.config.resultsTargetFrameName )
			{
				var f;
				if( parent )
					f = questagent.util.findLargestFrameDeep(parent);
				else
					f = questagent.util.findLargestFrameDeep(window);
				if( f && f.name )
					this.config.resultsTargetFrameName = f.name;
			}
			return this.config.resultsTargetFrameName;
		}

		/**
		Returns reference of frame for wait animation.
		If target window was not provided in configuration parameters, it will return
		reference of the largest frame.
		@function getWaitPanelTargetFrameRef
		@returns reference to target frame/window
		*/
		HtmlJSBasedModel.prototype.getWaitPanelTargetFrameRef = function()
		{
			if( !this.config.waitPanelTargetFrameRef )
			{
				var f;
				if( parent )
					f = questagent.util.findLargestFrameDeep(parent);
				else
					f = questagent.util.findLargestFrameDeep(window);
				if( f && f.frame )
					this.config.waitPanelTargetFrameRef = f.frame;
			}
			return this.config.waitPanelTargetFrameRef;
		}

		/**
		This function is responsible showing highlighted document.
		@function showDocumentHL
		@param {object} options - highlightiong options
		*/
		HtmlJSBasedModel.prototype.showDocumentHL = function( hlInfo ) // TODO? we're passing hlInfo although in context
		{
			log.debug( "Current document: " + window.location + " in " + window.name );

			log.debug( "Show highlighted: " + hlInfo.docLocation + " in " + this.getDocumentTargetFrameName() );

			var win = window.open(
				// open html document responsible for showing the highlighted document
				questagent.util.getQAHome(this.config) + 'hldoc.htm' +
				// pass location of original document in case something goes wrong
				'?document=' + escape(this.getAbsoluteUrlIfNotAlready( hlInfo.docLocation )) +
				// pass reference to questagent id that could be useful for window referencing
				'&ref=' + escape( questagent.id ) +
				// jump to the first highlighted match
				'#' + hlInfo.currentAnchor,
				this.getDocumentTargetFrameName() );
			this.config.windowWithDocument = win;
			win.focus();
		}

		/**
		This function is responsible showing document without highlighting.
		@function showDocumentNoHL
		@param {string} location - document location
		*/
		HtmlJSBasedModel.prototype.showDocumentNoHL = function( loc )
		{
			log.debug( "Show without highlighting: " + loc + " in " + this.getDocumentTargetFrameName() );
			var win = window.open( this.getAbsoluteUrlIfNotAlready(loc), this.getDocumentTargetFrameName() );
			this.config.windowWithDocument = win;
			win.focus();
		}

		/**
		Prepends baseURL if specified document location is not an absolute URL.
		@function getAbsoluteUrlIfNotAlready
		@param {string} location - document location
		*/
		HtmlJSBasedModel.prototype.getAbsoluteUrlIfNotAlready = function( loc )
		{
			if( loc.indexOf('file:') == 0 ||
				loc.indexOf('http:') == 0 || loc.indexOf('https:') == 0 ) {
				return loc;
			}
			return (this.config.baseURL + loc);
		}

		/**
		Shows document referenced by hit index without highlighting.
		@function showDocumentByIndexNoHL
		@param {int} hit - hit index
		*/
		HtmlJSBasedModel.prototype.showDocumentByIndexNoHL = function( hitIndex )
		{
			var loc = this.lastSearchResults.hits[hitIndex].location;
			this.showDocumentNoHL( loc );
		}

		/**
		Returns URL of QuestAgent search interface home.
		@function getBaseURL
		@returns base URL
		*/
		HtmlJSBasedModel.prototype.getBaseURL = function()
		{
			return this.config.baseURL;
		}

		/**
		This function is invoked in case Java was not found.
		By default shows alert with error message.
		@function onJavaNotFound
		@returns false if you don't want to proceed with applet initialization
		*/
		HtmlJSBasedModel.prototype.onJavaNotFound = function()
		{
			if( window.qagentBootstrap && window.qagentBootstrap.onJavaNotFound ) {
				return window.qagentBootstrap.onJavaNotFound();
			}

			alert('Search engine initialization script can not detect Java in your browser.\n' +
				'QuestAgent search applet requires Java 1.4.x or higher installed.\n' +
				'It is possible that Java is not installed on your system or has been disabled.\n\n' +
				'Will try to init search applet anyway...' );
			// return false if you don't want to proceed with applet initialization
			return true;
		}

		questagent.HtmlJSBasedModel = HtmlJSBasedModel;


		// TODO: DHtmlJSBasedModel is a work in progress
		var DHtmlJSBasedModel = function() {
			log.info('Initializing DHtmlJSBasedModel');
		};
		questagent.util.extendClass( DHtmlJSBasedModel, HtmlJSBasedModel );

		DHtmlJSBasedModel.prototype.search = function(searchParams,startHit) {
			log.debug("in DHtmlJSBasedModel.search()");

			if( searchParams )
				this.searchParams = searchParams;
			else
				log.debug("Reusing old search parameters (paging)");

			if( !this.searchParams )
			{
				log.error("Search parameters are not specified.");
				// FIXME report error to user
				return;
			}

			var waitFrameRef = this.getWaitPanelTargetFrameRef()
			if( waitFrameRef ) {
				log.debug('Found target for "wait" panel: ' + waitFrameRef.name);
				questagent.util.showPleaseWaitPanel(
					this.config.msgSearchInProgress, waitFrameRef, this.config );
			}

			if( startHit ) {
				this.searchParams.startHit = startHit;
			}

			this.appletWrapper.search(this.searchParams);
		}

		/**
		This function received starts results.
		@param {object} results - search results
		@function onSearchResults
		*/
		DHtmlJSBasedModel.prototype.onSearchResults = function(result) {
			log.debug("DHtmlJSBasedModel.onSearchResults");

			this.hlInfo = null;
			this.lastSearchResults = result;

			// build HTML segment with results and save with results
			result.html = this.buildResultsHTML(result);

			var myTextField = document.getElementById(this.config.targetElement);

			if(!myTextField) {
				alert("Not found element " + this.config.targetElement);
				return
			}

			myTextField.innerHTML = result.html;
		}

		DHtmlJSBasedModel.prototype.handleHighlightedDocument = function(hlPage)
		{
			questagent.util.hidePleaseWaitPanel(this.config.windowWithResults);

			// check is returned content actually error message
			if( hlPage.indexOf( "qagent_hl_msg:" ) == 0 )
			{
				var message = hlPage.substring( 14 ); // extract error message
				log.debug("Error on highlighting: " + message);
				this.handleHighlightingError(message,this.hlInfo);
				return;
			}

			log.debug("Received highlighted document.");

			// FIXME - how to set base?
			this.hlInfo.docHighlighted = '<base href="' + this.hlInfo.hlDocBase + '">' + hlPage;
			this.hlInfo.currentAnchor = "qa_anchor_1";

			this.showDocumentHL( this.hlInfo );
		}

		questagent.DHtmlJSBasedModel = DHtmlJSBasedModel;


		/**
		Search interface model that opens search results in popup window.
		@class questagent.PopupSearchWindowModel
		*/
		var PopupSearchWindowModel = function() {
			log.info('Initializing PopupSearchWindowModel');
		};

		/**
		Initializes this search interface model.
		@function init
		@param {object} config - configuration object
		*/
		PopupSearchWindowModel.prototype.init = function(cfg)
		{
			log.debug('PopupSearchWindowModel.init()');

			this.config = cfg;
			this.log = log;
			//this.lastSearchResults = null;
			//this.lastHighlightedDocIndex = null;


			if(window.qagentModel) {
				log.fatal("Only one search interface model can be defined per window/frame. " +
					"Found " + window.qagentModel + " so can't define " + this.id );
				return false;
			}
			window.qagentModel = this;

			var config = this.config; // this variable will be referenced in 'onLoad' handler bellow as 'this' can't be used

			Event.onDOMReady( function()
			{
				log.debug("QA home window loaded. Starting init...");

				var form = questagent.util.attachSearchFormHandlers(window, config);

				if( config.onLoad ) { // TODO replace with custom events
					log.debug('Invoke config.onLoad()');
					window.setTimeout( config.onLoad, 100, window.document);
				}

			}, window, true);
			log.debug("Added 'onDOMReady' event observer.");

			// if frame does not have name, assign one to it
			if( !window.name ) {
				window.name = questagent.id;
				log.debug('Assigned name "' + window.name + '" to window containing ' + window.URL );
			}

			// if document target window is not defined, set to current window
			if( !this.config.documentTargetFrameName ) {
				this.config.documentTargetFrameName = window.name;
			}
			log.debug('Document target window is: ' + this.config.documentTargetFrameName );


			log.info("Interface model initialized");
		}

		/**
		This function is invoked to start search.
		It doesn't start search directly but opens popup window passing search parameters.
		@function search
		@param {object} searchParams - object encapsulating search parameters
		*/
		PopupSearchWindowModel.prototype.search = function(searchParams) {
			log.debug('PopupSearchWindowModel.search()');

			var urlParams = '';

			if( searchParams.query )
				urlParams += "query=" + encodeURIComponent( searchParams.query ) + '&';

			if( searchParams.colls )
			{
				if( (typeof searchParams.colls) == 'string' ) {
					urlParams += "colls=" + encodeURIComponent( searchParams.colls ) + '&';
				}
				else {
					for( i = 0; i < searchParams.colls.length; ++i ) {
						urlParams += "colls=" + encodeURIComponent( searchParams.colls[i] ) + '&';
					}
				}
			}

			log.debug( 'Open popup with URL params: ' + urlParams );

			// open empty document just to get a reference to window
			var f = questagent.util.popupWindow(
				questagent.util.getQAHome(this.config) + 'empty.htm',
				window.name + 'Popup', // TODO could use config to decide between single and multi-instance?
				this.config.popupWidth, this.config.popupWidth, 'resizable=no' );

			if( !f ) {
				log.error("Failed to open popup window.");
				f = this.onPopupOpenFailed( popurl, window.name + 'Popup', "Can't initialize search.");
			}

			var popurl = window.questagent.util.getQAHome(this.config) + 'popup.htm?' + urlParams;
			if( f ) {
				// end now open search panel
				f.location.replace(popurl);
				f.focus();
			}
		}

		/**
		Invoked when script failed to open popup. Can report error or/and return reference to an alternative window.
		@function onPopupOpenFailed
		@param {string} url - document url
		@param {string} target - name of target popup window
		@param {string} message
		*/
		PopupSearchWindowModel.prototype.onPopupOpenFailed = function( url, target, message ) {
			alert("Failed to open popup window. " + message);

			var w = window.open( url, target );
			if( !w )
				alert( "Search engine failed to open popup window." );
			return w;
		}

		questagent.PopupSearchWindowModel = PopupSearchWindowModel;


		/**
		Search interface model that in popup window opens classic QuestAgent search applet.
		@class questagent.ExternalClassicAppletModel
		*/
		// @extends questagent.HtmlJSBasedModel ?? FIXME ?
		var ExternalClassicAppletModel = function() {
			log.info('Initializing ExternalClassicAppletModel');
		};

		ExternalClassicAppletModel.prototype = new HtmlJSBasedModel;
		function ExternalClassicAppletModelConstructor()
		{
			log.info('Initializing ExternalClassicAppletModel');
		    //questagent.HtmlJSBasedModel.call(this);  // Call super-class constructor (if desired)
		}
		ExternalClassicAppletModel.prototype.constructor = ExternalClassicAppletModelConstructor;

		/**
		Initializes this search interface model.
		@function init
		@param {object} config - configuration object
		*/
		ExternalClassicAppletModel.prototype.init = function(cfg)
		{
			log.debug('ExternalClassicAppletModel.init()');

			this.config = cfg;
			this.log = log;

			if(window.qagentModel) {
				log.fatal("Only one search interface model can be defined per window/frame. " +
					"Found " + window.qagentModel + " so can't define " + this.id );
				return false;
			}
			window.qagentModel = this;

			log.info('initialize wrapper');
			new questagent.SearchAppletWrapper(	this.config, this, this.getAppletInitHTML());
		}

		/**
		Connects this search interface model with applet wrapper.
		@function setAppletWrapper
		@param {object} wrapper - applet wrapper object
		*/
		ExternalClassicAppletModel.prototype.setAppletWrapper = function(wrapper) {
			this.appletWrapper = wrapper;

			log.debug('Subscribing to applet events. TODO?');
			// ...
		}

		/**
		Returns HTML segment that will be used to initialize search applet.
		@function getAppletInitHTML
		@returns string with HTML segment
		*/
		ExternalClassicAppletModel.prototype.getAppletInitHTML = function()
		{
			var size = questagent.util.getWindowSize();
			var padX = 45;
			var padY = 35;

			return questagent.util.getQAVisualAppletHTML( size.width - padX, size.height - padY, this.config);
		}

		questagent.ExternalClassicAppletModel = ExternalClassicAppletModel;

	})();

	// ======================== other questagent domain functions start ====

	/**@scope questagent */

	/**
	Desides on whether document can be highlighted or not.
	@function getHighlightingOptions
	@param {string} location - URI of document location
	@returns true if document should be highlighted, false otherwise
	*/
	questagent.canHighlight = function( location )
	{
		var loc2 = "" + location.toLowerCase();
		var canHL = false;

		if( ( loc2.indexOf( ".htm" ) + 4 ) == loc2.length )
			canHL = true;
		else if( ( loc2.indexOf( ".html" ) + 5 ) == loc2.length )
			canHL = true;

		if( canHL && loc2.indexOf( "../" ) != -1 )
		{
			log.warn( "Path to the document being highligted contains '../'. " +
				"This will be a problem if document is not in or bellow directory containing QuestAgent jar archive." );
			//canHL = false;
		}

		return canHL;
	}

	/**
	This function is invoked in the background to initialize applet wrapper.
	@function initSearchApplet
	*/
	questagent.initSearchApplet = function() {
		log.debug("questagent.initSearchApplet()");
		//log.debug("questagent.initSearchApplet for " + modelID);

		var model = window.qagentModel;
		if(!model) {
			log.fatal("Can't find QuestAgent search interface model object attached to window.");
			return;
		}

		new questagent.SearchAppletWrapper(	model.config, model, model.getAppletInitHTML());

		log.debug("Initialized");
	}

})();
}
